import React, { useEffect, useState } from 'react';
import swal from 'sweetalert';
import { useSelector, useDispatch } from 'react-redux';
import { fetchShortcuts, addShortcut, deleteShortcut, hideShortcut, unhideShortcut } from '@/app/lib/features/shortcuts/shortcutsSlice';
import { FaTrashAlt } from 'react-icons/fa';
import { RxCross2 } from "react-icons/rx";
import { FaEye } from "react-icons/fa6";
import { FaEyeSlash } from "react-icons/fa6";

export default function Shortcuts({ companyId, userId }) {
  const dispatch = useDispatch();
  const { shortcuts, shortcutdetails, status } = useSelector((state) => state.shortcuts);
  const [newShortcut, setNewShortcut] = useState('');
  const [shortcutType, setShortcutType] = useState(0); // 1 = Public, 0 = Private

  useEffect(() => {
    if (companyId && userId) {
      dispatch(fetchShortcuts({ companyId, userId }));
    }
  }, [companyId, userId, dispatch]);

  const handleAddShortcut = () => {
    const trimmedShortcut = newShortcut.trim();

    if (!trimmedShortcut) {
      return swal("Error", "Shortcut cannot be empty.", "error");
    }

    // Check if the shortcut already exists
    
    const isDuplicate = shortcuts.some((shortcut) => shortcut.text.toLowerCase() === trimmedShortcut.toLowerCase());
    if (isDuplicate) {
      return swal("Error", "This shortcut already exists.", "error");
    }

    dispatch(addShortcut({ userId, companyId, shortcut: trimmedShortcut, shortcutType }))
      .unwrap()
      .then(() => setNewShortcut('')) // Reset input field on success
      .catch(() => swal("Error", "Failed to add shortcut.", "error"));
  };

  const handleDeleteShortcut = (shortcutId) => {
    dispatch(deleteShortcut({ userId, companyId, shortcutId }))
      .unwrap()
      .catch(() => swal("Error", "Error deleting shortcut.", "error"));
  };

  const handleToggleVisibility = (shortcutId, isHidden) => {
    if (isHidden === 0) {
      dispatch(hideShortcut({ userId, shortcutId }))
        .unwrap()
        .then(() => {})
        .catch(() => swal("Error", "Failed to hide shortcut.", "error"));
    } else {
      dispatch(unhideShortcut({ userId, shortcutId }))
        .unwrap()
        .then(() => {})
        .catch(() => swal("Error", "Failed to unhide shortcut.", "error"));
    }
  };
  

  return (
    <div className="shortcuts-container">
      <div className="shortcut-input-wrapper">
        <textarea
          type="text"
          value={newShortcut}
          onChange={(e) => setNewShortcut(e.target.value)}
          placeholder="Add shortcut"
          disabled={status === 'loading'}
          className="shortcut-input"
        />
        <select
          value={shortcutType}
          onChange={(e) => setShortcutType(Number(e.target.value))}
          className="shortcut-type-select"
        >
          <option value={0}>Private</option>
          <option value={1}>Public</option>
        </select>
        <button
          onClick={handleAddShortcut}
          disabled={status === 'loading' || !newShortcut.trim()}
          className="shortcut-add-button"
        >
          Add Shortcut
        </button>
      </div>

      {status === 'loading' && <p>Loading...</p>}
      <table className="shortcuts-table">
  <thead>
    <tr>
      <th>No.</th>
      <th>Shortcut</th>
      <th>Created By</th>
      <th>Type</th>
      <th>Visibility</th>
      <th>Action</th>
    </tr>
  </thead>
  <tbody>
    {/* Private Shortcuts */}
    <tr>
      <td colSpan="6" className="category-header">Private Shortcuts</td>
    </tr>
    {shortcutdetails.filter((row) => row.shortcut_type === 0).length > 0 ? (
      shortcutdetails
        .filter((row) => row.shortcut_type === 0)
        .map((row, index) => (
          <tr key={row.id} className="shortcut-item">
            <td>{index + 1}</td>
            <td className="shortcut-text">{Array.isArray(row.shortcuts) ? row.shortcuts.join(", ") : row.shortcuts}</td>
            <td>{row.created_by}</td>
            <td>Private</td>
            <td>-</td>
            <td>
              {userId == row.user_id ? (
                <button
                  onClick={() => handleDeleteShortcut(row.id)}
                  disabled={status === "loading"}
                  className="shortcut-delete-button"
                >
                  <RxCross2 size={17} />
                </button>
              ) : '---'}
            </td>
          </tr>
        ))
    ) : (
      <tr>
        <td colSpan="6" className="no-shortcuts">No private shortcuts available.</td>
      </tr>
    )}

    {/* Public Shortcuts */}
    <tr>
      <td colSpan="6" className="category-header">Public Shortcuts</td>
    </tr>
    {shortcutdetails.filter((row) => row.shortcut_type === 1).length > 0 ? (
      shortcutdetails
        .filter((row) => row.shortcut_type === 1)
        .map((row, index) => (
          <tr key={row.id} className="shortcut-item">
            <td>{index + 1}</td>
            <td className="shortcut-text">{Array.isArray(row.shortcuts) ? row.shortcuts.join(", ") : row.shortcuts}</td>
            <td>{row.created_by}</td>
            <td>Public</td>
            <td>
              <span onClick={() => handleToggleVisibility(row.id, row.is_hidden)} className='pointer-cursor'>
                {row.is_hidden === 0 ? <FaEye size={16} /> : <FaEyeSlash size={16} />}
              </span>
            </td>
            <td>
              {userId == row.user_id ? (
                <button
                  onClick={() => handleDeleteShortcut(row.id)}
                  disabled={status === "loading"}
                  className="shortcut-delete-button"
                >
                  <RxCross2 size={17} />
                </button>
              ) : '---'}
            </td>
          </tr>
        ))
    ) : (
      <tr>
        <td colSpan="6" className="no-shortcuts">No public shortcuts available.</td>
      </tr>
    )}
  </tbody>
</table>

    </div>
  );
}
