import React, { useEffect, useState, useCallback } from "react";
import axios from "axios";
import Link from "next/link";
import AssignBrandModal from "./AssignBrandModal";
import { getCookie } from "cookies-next";
import { useSelector } from "react-redux";
import swal from "sweetalert";
import apiRequest from "@/app/lib/InterceptorAxios/axios";

const AllUsers = () => {
  const [users, setUsers] = useState([]);
  const [searchQuery, setSearchQuery] = useState("");
  const [filteredUsers, setFilteredUsers] = useState([]);
  const [uniqueCompanies, setUniqueCompanies] = useState([]);
  const [selectedCompany, setSelectedCompany] = useState("");
  const [selectedUserId, setSelectedUserId] = useState(null);
  const [selectedCompId, setSelectedCompId] = useState(null);
  const [UserData, setUserData] = useState();
  const [sortByActive, setSortByActive] = useState(true); // true for active users first, false for inactive users first

  const [showUserModal, setShowUserModal] = useState(false); // User modal visibility state
  const user = useSelector((state) => state.user.userInfo);
  const [showModal, setShowModal] = useState(false);
  const token = getCookie("token");
  const roleMapping = {
    1: "Super Admin",
    2: "Admin",
    3: "Sales",
    4: "Marketing",
  };
  const [userFormData, setUserFormData] = useState({
    name: "",
    pseudonym: "",
    email: "",
    password: "",
    status: 1,
    role: 3,
    designation: "",
    canCreateUsers: 0,
    canCreateBrand: 0,
    hrm_id: "",
  });
  // const openAddUserModel = async () => {
  //   setShowUserModal(true);
  //   const res = await axios-get(
  //           `${process.env.NEXT_PUBLIC_API_URL}/api/companies/${user.company_id}/brandnames`
  //         );

  // };
  const openAddUserModel = async () => {
    setShowUserModal(true);
    try {
      const response = await apiRequest(
        `/api/companies/${user.company_id}/brandnames`,
        "GET"
      );
      // Handle the response if needed
    } catch (error) {
      console.error("Error fetching brand names:", error.message);
    }
  };
  useEffect(() => {
    const fetchUserInfo = async () => {
      try {
        const response = await apiRequest(`/api/users/${user?.userId}`, "GET");
        setUserData(response);
      } catch (error) {
        console.error("Error fetching user information:", error.message);
      }
    };
    if (user.userId) {
      fetchUserInfo();
    }
  }, [user?.userId, token]); // Add user.userId and token to the dependency array

  useEffect(() => {
    // const fetchUsers = async () => {
    //   try {
    //     const response = await axios-get(
    //       `${process.env.NEXT_PUBLIC_API_URL}/api/users`,
    //       {
    //         headers: {
    //           Authorization: `Bearer ${token}`,
    //         },
    //       }
    //     );
    //     const allUsers = response.data;

    //     // Sort users dynamically based on the current sort order
    //     const sortedUsers = allUsers.sort((a, b) => {
    //       if (sortByActive) {
    //         return b.status - a.status; // Active users (status 1) come first
    //       } else {
    //         return a.status - b.status; // Inactive users (status 0) come first
    //       }
    //     });

    //     setUsers(sortedUsers);
    //     setFilteredUsers(sortedUsers);

    //     // Extract unique companies
    //     const companySet = new Set();
    //     const companies = sortedUsers.reduce((acc, user) => {
    //       if (!companySet.has(user.company_id)) {
    //         companySet.add(user.company_id);
    //         if (user.company_id && user.company_name) {
    //           acc.push({ id: user.company_id, name: user.company_name });
    //         }
    //       }
    //       return acc;
    //     }, []);

    //     setUniqueCompanies(companies);
    //   } catch (error) {
    //     console.error("Error fetching users:", error);
    //   }
    // };
    const fetchUsers = async () => {
      try {
        const response = await apiRequest("/api/users", "GET");
        const allUsers = response;

        // Sort users dynamically based on the current sort order
        const sortedUsers = allUsers.sort((a, b) => {
          if (sortByActive) {
            return b.status - a.status; // Active users (status 1) come first
          } else {
            return a.status - b.status; // Inactive users (status 0) come first
          }
        });

        setUsers(sortedUsers);
        setFilteredUsers(sortedUsers);

        // Extract unique companies
        const companySet = new Set();
        const companies = sortedUsers.reduce((acc, user) => {
          if (!companySet.has(user.company_id)) {
            companySet.add(user.company_id);
            if (user.company_id && user.company_name) {
              acc.push({ id: user.company_id, name: user.company_name });
            }
          }
          return acc;
        }, []);

        setUniqueCompanies(companies);
      } catch (error) {
        console.error("Error fetching users:", error.message);
      }
    };
    fetchUsers();
  }, [token, sortByActive]); // Re-fetch whenever the sort order changes

  const filterUsers = useCallback(() => {
    const results = users.filter((user) => {
      const matchesSearchQuery =
        user.name?.toLowerCase().includes(searchQuery.toLowerCase()) ||
        user.pseudonym?.toLowerCase().includes(searchQuery.toLowerCase()) ||
        user.email?.toLowerCase().includes(searchQuery.toLowerCase()) ||
        user.company_name?.toLowerCase().includes(searchQuery.toLowerCase());

      const matchesCompany =
        selectedCompany === "" || user.company_id === parseInt(selectedCompany);

      return matchesSearchQuery && matchesCompany;
    });

    setFilteredUsers(results);
  }, [searchQuery, selectedCompany, users]);

  useEffect(() => {
    filterUsers();
  }, [searchQuery, selectedCompany, filterUsers]);

  const handleSearch = (event) => {
    setSearchQuery(event.target.value);
  };

  const handleCompanyChange = (event) => {
    setSelectedCompany(event.target.value);
  };

  const handleAssignBrand = (userId, company_id) => {
    setSelectedUserId(userId);
    setSelectedCompId(company_id);
    setShowModal(true);
  };

  const handleCloseModal = () => {
    setShowModal(false);
    setSelectedUserId(null);
  };
  // const handleUserSubmit = async (e) => {
  //   e.preventDefault();

  //   if (user.userId) {
  //     try {
  //       const created_by = user.userId;
  //       const company_id = user.company_id;
  //       const updatedUserFormData = { ...userFormData, created_by, company_id };

  //       await axios-post(`${process.env.NEXT_PUBLIC_API_URL}/api/users`, updatedUserFormData, {
  //         headers: {
  //           Authorization: `Bearer ${token}`,
  //         },
  //       });

  //       setShowUserModal(false);
  //       setUserFormData({
  //         name: "",
  //         pseudonym: "",
  //         email: "",
  //         password: "",
  //         status: 1,
  //         role: 2,
  //         designation:'',
  //         canCreateUsers: 0,
  //         canCreateBrand: 0,
  //           hrm_id:''
  //       })
  //     // Show success alert if user is created successfully
  //     swal("Success!", "User created successfully.", "success");
  //       // Refresh user list after adding (if necessary)
  //     } catch (error) {
  //       if (error.response && error.response.status === 400 && error.response.data.error === 'Email already exists') {
  //         // Handle email already exists error
  //         console.error("Email already exists. Please use a different email.");
  //         // alert("Email already exists. Please use a different email.");
  //       // Show SweetAlert for "Email already exists" error
  //       swal("Oops!", "Email already exists. Please use a different email.", "error");
  //       } else {
  //         console.error("Error creating user:", error);
  //       }
  //     }
  //   } else {
  //     console.error("User not found");
  //   }
  // };
 const handleUserSubmit = async (e) => {
  e.preventDefault();

  if (user.userId) {
    try {
      const created_by = user.userId;
      const company_id = user.company_id;
      const updatedUserFormData = { ...userFormData, created_by, company_id };

      // API request to create the user
      const response = await apiRequest("/api/users", "POST", updatedUserFormData);
      const newUserId = response.id; // API only returns user ID

      if (!newUserId) {
        console.error("Failed to get new user ID.");
        return;
      }

      // Construct the new user object from form data
      const newUser = {
        ...userFormData,
        id: newUserId, // Assign the returned ID
        created_by,
        company_id,
      };

      setShowUserModal(false);
      setUserFormData({
        name: "",
        pseudonym: "",
        email: "",
        password: "",
        status: 1,
        role: 3,
        designation: "",
        canCreateUsers: 0,
        canCreateBrand: 0,
        hrm_id: "",
      });

      // Show success alert
      swal("Success!", "User created successfully.", "success");

      // Update users state by adding the new user
      setUsers((prevUsers) => {
        const updatedUsers = [...prevUsers, newUser];

        return updatedUsers.sort((a, b) => (sortByActive ? b.status - a.status : a.status - b.status));
      });

      // Update filtered users as well
      setFilteredUsers((prevFilteredUsers) => {
        const updatedFilteredUsers = [...prevFilteredUsers, newUser];

        return updatedFilteredUsers.sort((a, b) => (sortByActive ? b.status - a.status : a.status - b.status));
      });

      // Update unique companies if new user’s company is not already listed
      setUniqueCompanies((prevCompanies) => {
        const isCompanyExists = prevCompanies.some((c) => c.id === newUser.company_id);
        if (!isCompanyExists && newUser.company_id && user.company_name) {
          return [...prevCompanies, { id: newUser.company_id, name: user.company_name }];
        }
        return prevCompanies;
      });

    } catch (error) {
      if (error.message.includes("Email already exists")) {
        swal("Oops!", "Email already exists. Please use a different email.", "error");
      } else {
        console.error("Error creating user:", error.message);
      }
    }
  } else {
    console.error("User not found");
  }
};

  const handleUserInputChange = (e) => {
    const { name, value } = e.target;

    setUserFormData({
      ...userFormData,
      [name]: name === "brand_id" ? parseInt(value, 10) : value, // Convert brandid to integer
    });
  };

  const handleEnableDisableUser = async (id, status) => {
    try {
      // Show confirmation dialog
      const confirmed = await swal({
        title: "Are you sure?",
        text: "Do you want to change the user status?",
        icon: "warning",
        buttons: true,
        dangerMode: true,
      });

      if (confirmed) {
        const data = await apiRequest(`/api/users/${id}/status`, "PATCH", {
          status: status === 1 ? 0 : 1, // Toggle the status
        });

        if (data) {
          // Optionally, update the state or UI with the new status
          // Update the users state with the new status
          const newStatus = status === 1 ? 0 : 1;
          setFilteredUsers((prevUsers) =>
            prevUsers.map((user) =>
              user.id === id ? { ...user, status: newStatus } : user
            )
          );
          
        } else {
          console.error("Failed to update user status");
        }
      }
    } catch (error) {
      console.error("Error updating user status:", error);
    }
  };

  return (
    <>
      <div className="container-fluid company-header">
        <div className="comp-header-row flex justify-content-between align-items-center">
          <input
            type="text"
            className="form-control w-50"
            placeholder="Search Users"
            value={searchQuery}
            onChange={handleSearch}
          />
              {[1, 2, 4].includes(user.role)  && (
          <select
            className="form-control w-25"
            value={sortByActive ? "active" : "inactive"}
            onChange={(e) => setSortByActive(e.target.value === "active")}
          >
            <option value="active">Sort by Active Users</option>
            <option value="inactive">Sort by Inactive Users</option>
          </select>
              )}
          {UserData?.canCreateUsers === 1 && user?.role !== 1 && (
            <button className="btn" onClick={openAddUserModel}>
              Add User
            </button>
          )}
          {user.role === 1 && (
            <select
              className="form-control w-25"
              value={selectedCompany}
              onChange={handleCompanyChange}
            >
              <option value="">All Companies</option>
              {uniqueCompanies.map((company) => (
                <option key={company.id} value={company.id}>
                  {company.name}
                </option>
              ))}
            </select>
          )}
        </div>
      </div>
      <div className="total-users ">
        <p>Total Users: {filteredUsers.length}</p>
      </div>
      <div className="container-fluid table-container">
        <table className="table table-striped mt-4">
          <thead>
            <tr>
              <th>Name</th>
              <th>Pseudonym</th>
              <th>Designation</th>
              <th>Email</th>
              {/* Conditionally render the following headers based on user.role */}
              {[1, 2, 4].includes(user.role) && (
                <>
                  <th>Role</th>
                  <th>Company</th>
                  <th>Created by</th>
                  <th>Created At</th>
                  <th>Status</th>
                  <th>Actions</th>
                </>
              )}
            </tr>
          </thead>
          <tbody>
          {filteredUsers.length === 0 ? (
  <tr>
    <td colSpan="9">No Record found</td>
  </tr>
) : (
  filteredUsers.map((users) => {
    // If user.role == 3, skip users with status == 0
    if (user.role == 3 && users.status == 0) return null;

    return (
      <tr key={users?.id}>
        <td>{users?.name || "N/A"}</td>
        <td>{users?.pseudonym || "N/A"}</td>
        <td>{users?.designation || "N/A"}</td>
        <td>{users?.email || "N/A"}</td>
        {/* Conditionally render the following cells based on user.role */}
        {[1, 2, 4].includes(user.role) && (
          <>
            <td>{roleMapping[users?.role_id] || "Unknown"}</td>
            <td>{users?.company_name || "Unknown"}</td>
            <td>{users?.created_by_pseudonym || "N/A"}</td>
            <td>
              {users?.created_at
                ? new Date(users.created_at).toLocaleDateString()
                : "N/A"}
            </td>
            <td className={users?.status === 1 ? "active" : "inactive"}>
              {users?.status === 1 ? "Active" : "Inactive"}
            </td>
            <td>
              {user?.canCreateBrand === 1 && (
                <button
                  className="m-1 p-btn"
                  onClick={() =>
                    handleAssignBrand(users?.id, users?.company_id)
                  }
                >
                  Assign Brand
                </button>
              )}
            </td>
            <td>
              {user?.canCreateUsers === 1 && (
                <button
                  className="m-1 p-btn"
                  onClick={() =>
                    handleEnableDisableUser(users?.id, users?.status)
                  }
                >
                  {users?.status === 1 ? "Disable" : "Enable"}
                </button>
              )}
            </td>
          </>
        )}
      </tr>
    );
  })
)}

          </tbody>
        </table>
      </div>

      {showModal && user?.canCreateUsers == 1 && (
        <AssignBrandModal
          userId={selectedUserId}
          companyid={selectedCompId}
          onClose={handleCloseModal}
          role={user.role}
        />
      )}
      {showUserModal && (
        <div className="modal-overlay-comp">
          <div className="modal-brand">
            <div className="modal-header-comp">
              <h5 className="modal-title">Add User</h5>
              <button
                type="button"
                className="modal-close"
                onClick={() => setShowUserModal(false)}
              >
                &times;
              </button>
            </div>
            <div className="modal-body">
              <form onSubmit={handleUserSubmit}>
                <div className="form-group-row-2">
                  <div className="form-group">
                    <label htmlFor="name">Real Name</label>
                    <input
                      className="form-control"
                      type="text"
                      id="name"
                      name="name"
                      value={userFormData.name}
                      onChange={handleUserInputChange}
                      required
                    />
                  </div>
                  <div className="form-group">
                    {/* pseudonym is using as Pseudonym and name is using as name  */}
                    <label htmlFor="pseudonym">Pseudonym</label>
                    <input
                      type="text"
                      className="form-control"
                      id="pseudonym"
                      name="pseudonym"
                      value={userFormData.pseudonym}
                      onChange={handleUserInputChange}
                      required
                    />
                  </div>
                </div>
                <div className="form-group-row-2">
                  <div className="form-group">
                    <label htmlFor="email">Email</label>
                    <input
                      type="email"
                      className="form-control"
                      id="email"
                      name="email"
                      value={userFormData.email}
                      onChange={handleUserInputChange}
                      required
                    />
                  </div>
                  <div className="form-group">
                    <label htmlFor="password">Password</label>
                    <input
                      className="form-control"
                      type="password"
                      id="password"
                      name="password"
                      value={userFormData.password}
                      onChange={handleUserInputChange}
                      required
                    />
                  </div>
                  <div className="form-group">
                    <label htmlFor="designation">HRM ID:</label>
                    <input
                      type="text"
                      className="form-control"
                      id="hrm_id"
                      name="hrm_id"
                      value={userFormData.hrm_id}
                      onChange={handleUserInputChange}
                      required
                    />
                  </div>
                </div>
                <div className="form-group-row-2">
                  <div className="form-group flex-1">
                    <label htmlFor="designation">Designation:</label>
                    <input
                      type="text"
                      className="form-control"
                      id="designation"
                      name="designation"
                      value={userFormData.designation}
                      onChange={handleUserInputChange}
                      required
                    />
                  </div>
                  <div className="form-group flex-1">
                    <label htmlFor="status">Status</label>
                    <select
                      className="form-control"
                      id="status"
                      name="status"
                      value={userFormData.status}
                      onChange={handleUserInputChange}
                      required
                    >
                      <option value={1}>Active</option>
                      <option value={0}>Inactive</option>
                    </select>
                  </div>
                </div>
                <div className="form-group-row-2">
                  <div className="form-group flex-1">
                    <label htmlFor="role">Role</label>
                    <select
                      className="form-control"
                      id="role"
                      name="role"
                      value={userFormData.role}
                      onChange={handleUserInputChange}
                      required
                    >
                      <option value={2}>Admin</option>
                      <option value={3} selected>
                        Sales
                      </option>
                      <option value={4}>Marketing</option>
                      {/* <option value="client">Client</option> */}
                    </select>
                  </div>
                  <div className="form-group flex-1">
                    <label htmlFor="canCreateUsers">Can Create Users</label>
                    <select
                      className="form-control"
                      id="canCreateUsers"
                      name="canCreateUsers"
                      value={userFormData.canCreateUsers}
                      onChange={handleUserInputChange}
                      required
                    >
                      <option value={0}>No</option>
                      <option value={1}>Yes</option>
                    </select>
                  </div>
                </div>
                <div className="form-group">
                  <label htmlFor="canCreateBrand">Can Create Brand</label>
                  <select
                    className="form-control"
                    id="canCreateBrand"
                    name="canCreateBrand"
                    value={userFormData.canCreateBrand}
                    onChange={handleUserInputChange}
                    required
                  >
                    <option value={0}>No</option>
                    <option value={1}>Yes</option>
                  </select>
                </div>
                <button
                  type="submit"
                  className="btn submit-btn p-btn mt-3 w-100"
                >
                  Add User
                </button>
              </form>
            </div>
          </div>
        </div>
      )}
    </>
  );
};

export default AllUsers;
