'use client';

import React, { useState, useEffect } from 'react';
import apiRequest from "@/app/lib/InterceptorAxios/axios";
import { useSelector } from "react-redux";
import './alert.css';
import swal from 'sweetalert';
import flatpickr from 'flatpickr';
import 'flatpickr/dist/flatpickr.min.css';
import { getSocket } from '@/app/lib/socketManager';

const AlertsPage = () => {
  const [alerts, setAlerts] = useState([]);
  const [selectedAlert, setSelectedAlert] = useState(null);
  const [message, setMessage] = useState('');
  const [isActive, setIsActive] = useState(true);
  const [expiryDate, setExpiryDate] = useState('');
  const [targetedAlerts, setTargetedAlerts] = useState([]);
  const [oneTimeAlerts, setOneTimeAlerts] = useState([]);
  const [type, setType] = useState('one-time'); // New state for type
  const [socket, setSocket] = useState(null);
  const user = useSelector((state) => state.user.userInfo);
   
  const fetchAlerts = async () => {
    try {
      const response = await apiRequest('/api/alerts', 'GET');
      setAlerts(response.alerts || []);
    } catch (error) {
      console.error('Error fetching alerts:', error.message);
    }
  };
  useEffect(() => {
    const socketinst = getSocket();
    if (!socketinst) return;

    setSocket(socketinst);
 
  
     
  }, [])

  const fetchAlertsByCategory = async () => {
    try {
      const data = await apiRequest('/api/alerts/by-category', 'GET');
      
      setTargetedAlerts(data.targetedAlerts);
      setOneTimeAlerts(data.oneTimeAlerts);
    } catch (error) {
      console.error('Error fetching alerts by category:', error.message);
    }
  };

  useEffect(() => {
    fetchAlerts();
    fetchAlertsByCategory();
  }, []);

  const formatDate = (date) => {
    return new Date(date).toLocaleString('en-US', {
      day: 'numeric',
      month: 'short',
      year: 'numeric',
      hour: 'numeric',
      minute: 'numeric',
      second: 'numeric',
      hour12: true,
    });
  };

  const handleSelectAlert = (alert) => {
    setSelectedAlert(alert);
    setMessage(alert.msg);
    setIsActive(alert.is_active || false);
    setExpiryDate(alert.expired_at ? alert.expired_at.split('T')[0] : '');
    setType(alert.type || 'one-time'); // Set type when editing
  };

  const handleSubmit = async () => {
    try {
      if (!message.trim()) {
        swal("error","Message cannot be empty.","error");
        return;
      }
  
      const alertData = {
        msg: message,
        is_active: isActive,
        expired_at: isActive ? expiryDate : null,
        type: type,
        user_id: user.userId,
      };
  
      // Send alert via socket if it's a one-time alert
      if (type === "one-time") {
        const alertSend = {
          msg: message,
          is_active: isActive,
          expired_at: isActive ? expiryDate : null,
          alertType: type,
          user_id: user.userId,
        };
        
        // Emit alert via socket
        socket.emit("new-alert", alertSend);
        await apiRequest("/api/alerts", "POST", alertData);
        fetchAlerts();
        fetchAlertsByCategory();
        resetForm();
        return;
      }
  
      // Save alert via API for persistence
      if (selectedAlert) {
        // Update existing alert
        await apiRequest(`/api/alerts/${selectedAlert.id}`, "PUT", alertData);
      } else {
        // Create new alert
      
        let selectedDate = expiryDate ? new Date(expiryDate) : null;
        
        
        if (!selectedDate) {
          swal('Error', 'Expiry date is required to activate the alert.', 'error');
          return;
        }
      
        await apiRequest("/api/alerts", "POST", alertData);
      }
  
      // Reset form and fetch updated alerts
      resetForm();
      fetchAlerts();
      fetchAlertsByCategory();
    } catch (error) {
      console.error("Error submitting alert:", error.message);
    }
  };

  const handleSendAlert = (alert)=>{
    const alertSend = {
      msg: alert.msg,
      is_active: alert.is_active,
      expired_at: isActive ? expiryDate : null,
      alertType: alert.type,
      user_id: user.userId,
    };
    
    socket.emit("new-alert", alertSend);
  }
  const handleToggleAlertState = async (alert) => {
    try {
      const newStatus = !alert.is_active;
      
      const apiEndpoint = newStatus
        ? `/api/alerts/${alert.id}/publish`
        : `/api/alerts/${alert.id}/deactivate`;

      if (newStatus) {
        let selectedDate = alert.expired_at ? new Date(alert.expired_at) : null;
  
        if (alert.type !== 'one-time') {
          const dateInput = document.createElement('input');
          dateInput.type = 'text';
  
          // Configure flatpickr with minDate set to today
          flatpickr(dateInput, {
            defaultDate: selectedDate,
            dateFormat: 'Y-m-d H:i', // Include hours and minutes in the format
            minDate: 'today', // Prevent selecting past dates
            defaultHour: 23, // Set default hour to 23 (11 PM)
            defaultMinute: 59, // Set default minute to 59
            onChange: (selectedDates) => {
              selectedDate = selectedDates[0];
          
              // Append seconds manually (if needed)
              if (selectedDate) {
                const updatedDate = new Date(selectedDate);
                updatedDate.setSeconds(59); // Set seconds to 59
                selectedDate = updatedDate; // Update selected date with seconds
              }
            },
          });
          
  
          const result = await swal({
            title: "Select Expiry Date",
            content: dateInput,
            buttons: {
              cancel: true,
              confirm: { text: "Activate", closeModal: false },
            },
          });
  
          if (!selectedDate) {
            swal('Error', 'Expiry date is required to activate the alert.', 'error');
            return;
          }
        }
  
        await apiRequest(apiEndpoint, 'PUT', { expired_at: selectedDate ? selectedDate.toISOString().split('T')[0] : null });
        swal('Success', 'Alert activated successfully!', 'success');
      } else {
        await apiRequest(apiEndpoint, 'PUT', { expired_at: null });
        swal('Success', 'Alert deactivated successfully!', 'success');
      }
  
      fetchAlertsByCategory();
    } catch (error) {
      console.error('Error toggling alert state:', error.message);
      swal('Error', 'There was an error toggling the alert state.', 'error');
    }
  };
  const handleDelete = async (alert) => {
    const apiEndpoint = `/api/alerts/${alert.id}/delete`;
  
    try {
      // Make the API request
      const response = await apiRequest(apiEndpoint, 'DELETE', { expired_at: null });
  
      if (response.message) {
        // swal('Success', 'Alert Deleted successfully!', 'success');
        if(alert.alertType=='one-time'){
          setOneTimeAlerts((prevAlerts) => prevAlerts.filter((a) => a.id !== alert.id))
        }
        else{
          setTargetedAlerts((prevAlerts) => prevAlerts.filter((a) => a.id !== alert.id))
        }
       fetchAlertsByCategory()
      } else {
        const errorData = await response.json();
        swal('Error', errorData.message || 'Failed to delete alert', 'error');
      }
    } catch (error) {
      console.error('Error deleting alert:', error);
      swal('Error', 'An unexpected error occurred while deleting the alert', 'error');
    }
  };
  
  const handleSetAlertActiveChange = (e) => {
    setIsActive(e.target.checked);
  };

  const resetForm = () => {
    setSelectedAlert(null);
    setMessage('');
    setIsActive(true);
    setExpiryDate('');
    setType('one-time'); // Reset type
  };

  return (
    <div className="alert-container">
      <h1 className="alert-header">Alerts</h1>

      <div className="alert-form">
  <div className="alert-row">
    <textarea
      value={message}
      onChange={(e) => setMessage(e.target.value)}
      placeholder="Enter alert message"
      className="alert-input"
    ></textarea>
    <button onClick={handleSubmit} className="alert-button">
      {selectedAlert ? "Update Alert" : "Create Alert"}
    </button>
  </div>
  <div className="alert-row">
   
    <select
      value={type}
      onChange={(e) => setType(e.target.value)}
      className="alert-select"
    >
      <option value="one-time">One-time</option>
      <option value="targeted">Targeted</option>
    </select>
    {type === "targeted" && (
      <label className="alert-checkbox">
        <input
          type="checkbox"
          checked={isActive}
          onChange={handleSetAlertActiveChange}
        />
        Active Now
      </label>
    )}
    <span style={{fontSize:"16px"}} className='alert-checkbox'>expiry</span>
    {isActive && type === "targeted" && (
      <input
        type="date"
        value={expiryDate}
        onChange={(e) => setExpiryDate(e.target.value)}
        min={new Date().toISOString().split("T")[0]} // Set minimum date to today
        className="alert-date"
      />
    )}
  </div>
</div>

      {/* Targeted and One-time Alerts Tables */}
      <h2 className="alert-subheader">Targeted Alerts</h2>
      <table className="alert-table">
        <thead>
          <tr>
            <th>Message</th>
            <th>Type</th>
            <th>Expiry Date</th>
            <th>Action</th>
          </tr>
        </thead>
        <tbody>
          {targetedAlerts.map((alert) => (
            <tr key={alert.id}>
              <td>{alert.msg}</td>
              <td>{alert.type}</td>
              <td>{alert.expired_at ? formatDate(alert.expired_at) : 'No expiry'}</td>
              <td>
  <div className="button-group">
    {alert.type === "targeted" && alert.is_active === 1 && (
      <button
        className="btn text-light alert-toggle-btn-deactivate bg-danger"
        onClick={() => handleToggleAlertState(alert)}
      >
        Deactivate
      </button>
    )}
    {alert.type === "targeted" && alert.is_active === 0 && (
      <button
        className="alert-toggle-btn btn p-btn"
        onClick={() => handleToggleAlertState(alert)}
      >
        Activate
      </button>
    )}
    <button
      className="alert-toggle-btn btn p-btn"
      onClick={() => handleDelete(alert)}
    >
      Remove
    </button>
  </div>
</td>

            </tr>
          ))}
        </tbody>
      </table>

      <h2 className="alert-subheader">One-time Alerts</h2>
      <table className="alert-table">
        <thead>
          <tr>
            <th>Message</th>
            <th>Type</th>
            <th>Expiry Date</th>
            <th>Action</th>
          </tr>
        </thead>
        <tbody>
          {oneTimeAlerts.map((alert) => (
            <tr key={alert.id}>
              <td>{alert.msg}</td>
              <td>{alert.type}</td>
              <td>{alert.expired_at ? formatDate(alert.expired_at) : 'No expiry'}</td>
              <td>
              <div className="button-group">
                <button
                  className="alert-toggle-btn btn p-btn"
                  onClick={() => handleSendAlert(alert)}
                >
                  {/* {alert.is_active ? 'Deactivate' : 'Activate'} */}
                  Send Alert

                </button>
                <button
                  className="alert-toggle-btn btn p-btn"
                  onClick={() => handleDelete(alert)}
                >
                  Remove
                </button>
                </div>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
};

export default AlertsPage;
